#include-once
; feel free to tap into this in your apps, I do.

#cs
  corz gui window animation effects v0.2.1
  functions to simplify window open/close animations

  (c) cor + corz.org 2006->tomorrow!


	function list..
	[see the functions themselves for more detailed usage instructions]

		corz_GUIAnimateOpen( [window handle[, string/array{animation codes}[, string{ini file path}[, string{ini section}]]]])

		corz_GUIAnimateClose( [window handle[, string/array{animation codes}[, string{ini file path}[, string{ini section}]]]])

		corz_GUIFadeIn( window handle )

		corz_GUIFadeOut( window handle )

		corz_GUIGetFX( [string{fx setting} [, string{path to ini file} [, string{ini section} [, string{ini key}]]]])

			[private] corz_ValidateFXCodes( string/array{animation codes}[, string{ini file path}[, string{ini section}]]])


	In most cases, you will only need to interact with the two main functions:
	corz_GUIAnimateOpen() and corz_GUIAnimateClose(). They will access the other
	functions in the background, saving and restoring settings to and from your
	specified ini file, performing the desired window animations, and so on.

	You can also interact directly and easily with corz_GUIFadeIn() and
	corz_GUIFadeOut(), and even corz_GUIGetFX(), if you feel the need.



; The functions..
;

  corz_GUIAnimateOpen()
  Animate the opening of a window.

	Feed it the window handle and (optionally) some animation codes (which can
	be a simple string, like "slide right" - which means it slide out of the
	right hand side). Optionally feed it the path to your ini file, and the
	section of that ini to use for (re)storing settings..

		corz_GUIAnimateOpen( [window handle[, string/array{animation string/codes}[, string{path to ini} [, string{section name}]]]])

	Simplest usage..

		corz_GUIAnimateOpen($my_gui)

	Which is the same as..

		corz_GUIAnimateOpen($my_gui, "Slide Right")

	"Slide Right" being the default gui animation (remember, it was designed for
	wee tools, that most often, at least on my system, find themselves in the
	far-right corners).

		corz_GUIAnimateOpen($my_gui, "Slide Top Right")

	Is also nice.

	Available animation strings..
	[strings are case-insensitive; supplied strings & their case are retained in your ini]

		To fade in/out							use:	"fade" ; note, this doesn't use a dll call.
		To Slide in & out from the left			use:	"slide left", "left", "Slide L", or "L"
		To Slide in & out from the right		use:	"slide right", "right", "Slide R", "Slide", or "R"	(default fx)
		To Slide in & out from the top			use:	"slide top", "Slide T", "Top", or "T"
		To Slide in & out from the bottom		use:	"slide bottom", "Slide B", "Bottom",  or "B"

		To Slide in & out from the top-left		use:	"Slide Top Left", "Top Left", "top-left", "tl", "diagonal tl", "d"
		To Slide in & out from the top-right	use:	"Slide Top Right", "Top Right", "top-right", "tr", "diagonal tr"
		To Slide in & out from the bottom-right	use:	"Slide Bottom Right", "Bottom Right", "bottom-right", "br", "diagonal br"
		To Slide in & out from the bottom-left	use:	"Slide Bottom Left", "Bottom Left", "bottom-left", "bl", "diagonal bl"

		To Explode/implode						use	:	"explode/implode", "Explode", "Implode", or "X"

		To Disable Window Animation FX			use:	"Disable window animation", "Disabled", or "-"

	If you want your user's animation settings to be stored and retrieved
	between launches, but don't want to write the code for that, simply supply
	the optional 3rd and 4th parameters; the full path to an ini file, and the
	name of the section within that file..

		corz_GUIAnimateOpen($my_gui, "bottom", "test.ini", "my fx")

	Which will open the window, sliding it up from the bottom, and save this
	preference to "test.ini", in the section [fx]. It's no problem to use
	your main ini section, in that case a key named "gui_fx" will be written
	with the current animation string.

	If you want to save some animation setting, but not actually animate the
	window (for instance, when a user selects a preference from a menu), simply
	send an empty string for the window handle, and specify some valid ini
	parameters..

		corz_GUIAnimateOpen("", "bottom", "test.ini", "my fx")

	It's slightly non-standard, I know, but it saves you ever having to interact
	with corz_GUIGetFX() directly, which I thought you might like.

	Used in this manner, corz_GUIAnimateOpen() will return valid animation
	codes, which you may or may not wish to use later. See corz_GUIGetFX()
	for a larger explanation of how these codes might be used. One reason why
	you might use these codes is that they are slightly faster to operate.

	GUI fx doesn't create any global variables*, so it's a good idea to make
	$my_codes (or whatever you call it) global in your application, available
	for whenever you need to animate a window.

	You will probably want to do that when you first initialize your
	window..

		global $my_codes = corz_GUIAnimateOpen($my_gui, "", $ini_path, $ini_section)

	If the animation string is left blank "", settings will be retrieved from
	the ini file, if one is specified (and created if not). In either case,
	corz_GUIAnimateOpen() will return the valid animation codes for later use.
	If the ini's gui_fx string is invalid or simply doesn't exist, or you
	didn't specify valid ini paramters, the default setting will be applied.

	As well as returning the valid animation codes array, corz_GUIAnimateOpen()
	sets an environment variable (the name of the current "gui_fx" setting) to
	the simple string version of the animation setting, as saved in the ini
	file. You can use this to set menu selections, checkbox items, or whatever.

		GUICtrlSetData($combo, EnvGet('gui_fx'))

	Although it's usually 'gui_fx', you may have multiple window animations
	running concurrently, so it could be something else. See corz_GUIGetFX() for
	details on how to override this value. Note: these environment variable are
	destroyed when you quit the application.

	Whatever happens, if you specify valid ini parameters, corz_GUIAnimateOpen()
	will attempt to save the current setting to the specified ini at the end of
	its operation.


	Usage examples..

		Retrieve saved setting from the ini, and animate the window using those
		setting. This is most useful when the application first launches..

			corz_GUIAnimateOpen($my_gui, "", "test.ini", "my fx")


		Save the setting "Slide Top" to the ini file, perform no animation. This is
		most useful when the user selects a preference from a menu or some other
		selector..

			corz_GUIAnimateOpen("", "Slide Top", "test.ini", "my fx")

		"Slide Top" could come from anywhere..

			corz_GUIAnimateOpen("", TrayItemGetText(@TRAY_ID), $ini_path, $my_app)


		Open the window, animating it up from the bottom, and save that setting
		to the ini file. Perhaps the user okayed a preference dialog, and now
		the main gui is re-opening with updated setting..

			corz_GUIAnimateOpen($my_gui, "bottom", "test.ini", "my fx")


	Examples of how to automate implementation of GUI fx in your code..

		The long-winded versions of the animation strings are most useful; they
		can be used directly in menus, which greatly simplifies the code
		required to implement user-selectable window animations.

		It's fairly easy to animate directly from either saved ini settings, or
		user-selected preferences, or just save and retrieve settings, or both,
		and keep both those things synchronized in the process.

		Giving the user control is always a good thing (particularly when it's
		in layers of expertise), and personally, I like my tools to attach to
		specific areas of my desktop. The bonus of having these tools "slide
		out" of a particular area of the desktop only adds to the user's feeling
		of control.

		So let's say we wanted to create a tray menu for the user to select an
		animation style for your (their) program. The trick is, when we create
		the menu items, we use the actual text of the animation strings.

		Here's a complete tray menu you can rip right off the page..

			TrayCreateItem("")
			global $TrayMenuWinAnim = TrayCreateMenu("Window animation..")

			; we could re-use this array to create other controls (see the included example scripts)..
			local $win_anims[11] = ["Slide Left", "Slide Right", "Slide Top", "Slide Bottom", _
									"Slide Top Left", "Slide Top Right", "Slide Bottom Right", "Slide Bottom Left", _
									"Explode/Implode", "Fade", "Disable window animation"]

			global $menu_items_win_anim[11]
			for $i = 0 to 10
				$menu_items_win_anim[$i] = TrayCreateItem($win_anims[$i], $TrayMenuWinAnim, -1, 1)
				TrayItemSetOnEvent(-1, "MenuSetWinAnim")
			next

			for $i in $menu_items_win_anim
				if TrayItemGetText($i) = $my_fx then TrayItemSetState($i, $TRAY_CHECKED)
			next
			TrayCreateItem("")

		We can then use this exact same text to set the animation fx in the
		called function. Our fictional MenuSetWinAnim() function would be a
		skinny thing that might look very much like this..

			func MenuSetWinAnim()
				corz_GUIAnimateOpen("", TrayItemGetText(@TRAY_ID), "test.ini", "my fx")
			endfunc

		Or, if you wanted to grab the animation codes array for later use (a
		good idea, and while we're with the good ideas, you would of course be
		using variables of some sort)..

			func MenuSetWinAnim()
				global $my_codes = corz_GUIAnimateOpen("", TrayItemGetText(@TRAY_ID), $ini_path, $section)
			endfunc

		Then, whenever you need to animate any window, you simply feed the codes
		to the animate functions, like so..

			corz_GUIAnimateOpen($my_gui, $my_codes)

		Even if you didn't get the codes, or even know the animation string, you
		could still perform the window's current animation, like so..

			corz_GUIAnimateOpen($my_gui, EnvGet('gui_fx'))


		All parameters are optional. You can retrieve settings, or save settings,
		or perform a window animation, or any or all of those things at once.

		Simply put; If you want an animation, include the gui handle. If you
		don't want an animation, don't. If you want to save or retrieve
		settings, include ini parameters. If you want to save settings, include
		the animation string, and if you want to retrieve settings, don't.

		See the included "corz_gui_fx [tray+combo example].au3" and "corz_gui_fx
		[combo example].au3" for simple working demos of gui fx.

#ce
func corz_GUIAnimateOpen($some_gui="", $GUIFX="", $ini_file="", $ini_section="")

	if not $some_gui then
		return corz_GUIGetFX($GUIFX, $ini_file , $ini_section)
	else
		if not IsArray($GUIFX) then
			$GUIFX = corz_ValidateFXCodes($GUIFX, $ini_file, $ini_section)
		endif

		switch $GUIFX[1]
			case "-" ; animation has been disabled
				GUISetState(@SW_SHOW, $some_gui)
				return
			case "fade" ; we won't use the dll call version of the fade, it's nasty.
				corz_GUIFadeIn($some_gui)
			case else
				DllCall("user32.dll", "int", "AnimateWindow", "hwnd", $some_gui, "int", 200, "long", $GUIFX[1])
				Sleep(50)
				GUISetState(@SW_SHOW, $some_gui)
		endswitch
		return $GUIFX
	endif
endfunc

; corz_GUIAnimateClose()
; Animate the closing of a window.
;
; Usage as corz_GUIAnimateOpen() (above)
;
; Note: At the end of this function, the window is in a HIDDEN state.
;
func corz_GUIAnimateClose($some_gui="", $GUIFX="", $ini_file="", $ini_section="")

	; I'd love to slip this in somehow, to release the memory (at least, in appearance)
	; but there's just no way to do it without flashing the window, afaik.
	; GUISetState(@SW_MINIMIZE, $some_gui) ; I tried *everything* :/

	if not $some_gui then
		return corz_GUIGetFX($GUIFX, $ini_file, $ini_section)
	else
		if not IsArray($GUIFX) then
			$GUIFX = corz_ValidateFXCodes($GUIFX, $ini_file, $ini_section)
		endif
		switch $GUIFX[1]
			case "-"
				GUISetState(@SW_HIDE, $some_gui)
				return
			case "fade"
				GUISetState(@SW_SHOW, $some_gui)
				corz_GUIFadeOut($some_gui)
			case else
				GUISetState(@SW_SHOW, $some_gui)
				DllCall("user32.dll", "int", "AnimateWindow", "hwnd", $some_gui, "int", 200, "long", $GUIFX[2])
;				Sleep(50)
				GUISetState(@SW_HIDE, $some_gui)
		endswitch
		return $GUIFX
	endif
endfunc


; fade-in..
;
; you can access these two directly, too.
func corz_GUIFadeIn($some_gui)
	WinSetTrans($some_gui, "", 0)
	GUISetState(@SW_SHOW, $some_gui)
	for $i = 0 to 240 step 15
		WinSetTrans($some_gui, "", $i)
		Sleep(5)
	next
	WinSetTrans($some_gui, "", 255)
endfunc

; and out
; sadly, setting the transparency away from max causes the window to flicker briefly.
; but that's still better than the dll call version, which can basically disable the
; parent gui permanently.
func corz_GUIFadeOut($some_gui)
	for $i = 240 to 15 step - 15
		Sleep(5)
		WinSetTrans($some_gui, "", $i)
	next
	GUISetState(@SW_HIDE, $some_gui)
endfunc


; This enables you to send everything directly to corz_GUIAnimateOpen/Close()
;
;
; if we are already working with the animation codes, go away quickly, get on
; with the animaton. if we need to create valid codes, or it's a simple disable
; or fade, invoke corz_GUIGetFX().

func corz_ValidateFXCodes(ByRef $GUIFX, $ini_file="", $ini_section="")
	if IsArray($GUIFX) then return
	$GUIFX = corz_GUIGetFX($GUIFX, $ini_file, $ini_section)
	return $GUIFX
endfunc


#cs
	corz_GUIGetFX( [string{fx setting} [, string{path to ini file} [, string{ini section} [, string{ini key}]]]])

	All parameters are optional.

	This function validates the supplied animation string (retrieving it from
	the ini if no code/string is supplied), and returns an array of valid
	animation codes for later use, possibly by corz_GUIAnimateOpen/Close()

	If the supplied string is invalid, corz_GUIGetFX() makes it valid. Finally,
	if valid ini parameters were specified, the new animation string is saved to
	the ini file..


	Feed it the path and section name of the ini file you would like to use to
	(re)store the settings, and your chosen setting (or omit that, if you want
	corz_GUIGetFX() to grab it from the ini file, or create it anew, if no
	ini setting exists). See corz_GUIAnimateOpen(above) for a list of the
	available animation strings.

	As with corz_GUIAnimateOpen(), the long-winded versions can be used directly
	in menus..

			$menu_item[2] = TrayCreateItem("Slide Right", $TrayMenuWinAnim, -1, 1)
			TrayItemSetOnEvent(-1, "MenuSetWinAnim")

	Inside our fictional MenuSetWinAnim() function we would need to do no more
	than..

		$my_codes = corz_GUIGetFX(TrayItemGetText(@TRAY_ID), $ini_path, $section)

	This does ezactly the same thing, but *slightly* slower..

		$my_codes = corz_GUIAnimateOpen("", TrayItemGetText(@TRAY_ID), $ini_path, $section)

	Either way, if the user selects the item with the text "Slide right" (or
	whatever), corz_GUIGetFX() returns the valid animation codes into
	$my_codes (now an array) for later use. Then, when you need to animate any
	window, you simply feed this to the animate functions, like so..

		corz_GUIAnimateOpen($my_gui, $my_codes)

	and corz_GUIAnimateOpen() will choose the correct code ($my_codes[1]) and
	perform the required animation. $my_codes can hang around your application
	for whenever you need to animate a window (i.e. make it global).

	Of course, you can also call the open and close functions to perform these
	operations, but I figured you may want to access them directly, too.

	Whichever way you go, to you, the coder, and to the user, it's just a bunch
	of cute strings.

	Usage:

	The simplest form of usage is..

		$my_var = corz_GUIGetFX()

	Which will return an array of animation codes for "Slide Right" (the default
	animation). If you want to specify the animation, but don't need to (re)store
	any settings, use..

		$my_var = corz_GUIGetFX("bottom")	; or whatever

	If you want to specify an ini file for the settings, but leave everything
	else to corz_GUIGetFX(), use..

		$my_var = corz_GUIGetFX("", "C:\ini.file", "section name")

	and corz_GUIGetFX() will retrieve the last saved settings and apply them.
	If no settings were found, it creates them. At the end, it saves them.

	If you are trigerring all this from menus (as in above example), or similar,
	there's no need to do anything else, corz_GUIGetFX() will keep the ini
	file constantly updated with the latest settings.

	Another reason why you might want to interact directly with
	corz_GUIGetFX() is to override the name of the ini key/environment variable.
	Usually it is "gui_fx", but you might be animating lots of windows within a
	single application, each with individual settings.

	Then you could access the simple animation string of any window by getting
	the environment variable with its name; i.e..

		GUICtrlSetData($combo_img_tools, EnvGet('image_tools'))

	or whatever. And although you could also do this..

		corz_GUIAnimateOpen($gui_img_tools, EnvGet('image_tools'))

	you would most likely be working with the codes directly, having grabbed
	them already.

#ce
func corz_GUIGetFX($fx="", $fx_ini_path="", $fx_ini_section="corz gui fx", $fx_win = "gui_fx")
	if $fx_ini_path and $fx = "" then
		$fx = IniRead($fx_ini_path, $fx_ini_section, $fx_win, "Slide right")
	endif

	switch $fx
		case "-", "Disabled", "disable", "Disable window animation"
			local $GUIFX[3] = [2, "disable", "disable"] ; it don't like "-" here :/

		case "fade"
			local $GUIFX[3] = [2, "fade", "fade"]

		case "Slide Left", "Left", "Slide L", "l"
			local $GUIFX[3] = [2, '0x00040001', '0x00050002'] ; hex looks better darker (in my color scheme ;o)

		case "Slide Top", "Slide T", "Top", "t"
			local $GUIFX[3] = [2, '0x00040004', '0x00050008']

		case "Slide Right", "Right", "Slide R", "Slide", "r"
			local $GUIFX[3] = [2, '0x00040002', '0x00050001']

		case "Slide Bottom", "Slide B", "Bottom", "b"
			local $GUIFX[3] = [2, '0x00040008', '0x00050004']

		case "Slide Top Left", "Top Left", "top-left", "tl", "diagonal tl", "d"
			local $GUIFX[3] = [2, '0x00040005', '0x0005000a']

		case "Slide Top Right", "Top Right", "top-right", "tr", "diagonal tr"
			local $GUIFX[3] = [2, '0x00040006', '0x00050009']

		case "Slide Bottom Right", "Bottom Right", "bottom-right", "br", "diagonal br"
			local $GUIFX[3] = [2, '0x0004000a', '0x00050005']

		case "Slide Bottom Left", "Bottom Left", "bottom-left", "bl", "diagonal bl"
			local $GUIFX[3] = [2, '0x00040009', '0x00050006']

		case  "x", "Implode/Explode", "Explode/Implode", "Explode", "Implode"
			local $GUIFX[3] = [2, '0x00040010', '0x00050010']

		case "" ; deleted in the ini!
			continuecase
		case else
			$fx = "Slide Right"
			local $GUIFX[3] = [2, '0x00040002', '0x00050001']
	endswitch
	EnvSet($fx_win, $fx)
	if $fx_ini_path then IniWrite($fx_ini_path, $fx_ini_section, $fx_win, $fx)
	return $GUIFX
endfunc

